import { useEffect, useState } from 'react';
import { Button, IconButton, Box, Tooltip } from '@mui/material';
import { ContentCopy as ContentCopyIcon, LightbulbOutline as LightbulbOutlineIcon } from '@mui/icons-material';
import { useDispatch, useSelector } from 'react-redux';
import { makeStyles } from 'tss-react/mui';
import AceEditor from "react-ace";
import "ace-builds/src-noconflict/mode-json";
import "ace-builds/src-noconflict/theme-github";
import "ace-builds/src-noconflict/theme-tomorrow_night";
import { aceTheme } from '../utils';
import { handleCopy } from '../../../utils';
import { selectJsonConfig } from '../store/cmdbSlice/selectors';
import { getJSONConfig } from '../store/cmdbSlice/thunks';
import Dialog from '../../common/Dialog';
import useDarkMode from 'src/hooks/useDarkTheme';

interface ShowRenderedJsonProps {
  identifier: string;
}

export const useStyles = makeStyles()((theme) => ({
    editor: {
        borderRadius: 2,
        border: `1px solid ${theme.palette.border.main}`,
        width: '100%'
    },
    editorContainer: {
        position: 'relative'
    },
    copyButton: {
        position: 'absolute',
        top: 8,
        right: 8,
        zIndex: 10
    }
}));

const ShowRenderedJson: React.FC<ShowRenderedJsonProps> = ({ identifier }) => {
    const [open, setOpen] = useState<boolean>(false);
    const { classes } = useStyles();
    const jsonConfig = useSelector(selectJsonConfig);
    const dispatch = useDispatch();

    const handleOpen = () => setOpen(true);
    const handleClose = () => setOpen(false);

    useEffect(() => {
        if (!open || !identifier) return;
        dispatch(getJSONConfig({ identifier }));
    }, [identifier, open, dispatch]);

    const handleCopyJson = () => {
        if (jsonConfig) {
            handleCopy(jsonConfig);
        }
    };

    const theme = aceTheme(useDarkMode());
    
    return (
        <>
            <Button onClick={handleOpen} variant="outlined" size="small">
                Show JSON
            </Button>
            <Dialog
                open={open}
                fullWidth
                size={false}
                width={750}
                onClose={handleClose}
                title="Host JSON data"
                actionButtons={
                    <Button onClick={handleClose} variant="outlined" color="inherit" sx={{ mr: 1 }}>
                        Close
                    </Button>
                }
            >
                <Box>
                    This JSON includes data from both shared groups and host specific data. The host specific data files are in the &nbsp;
                    <code>/var/cfengine/cmdb</code> folder on the hub and <code>/var/cfengine/data</code> folder on the client.
                </Box>
                <Box className={classes.editorContainer} sx={{ mt: 2 }}>
                    <AceEditor
                        mode="json"
                        theme={theme}
                        readOnly
                        value={jsonConfig || ''}
                        tabSize={2}
                        style={{
                            width: '100%'
                        }}
                        setOptions={{
                            maxLines: 16,
                            minLines: 16,
                            useWorker: false
                        }}
                        className={classes.editor}
                    />
                    <Tooltip title="Copy to clipboard" placement="left">
                        <IconButton
                            className={classes.copyButton}
                            onClick={handleCopyJson}
                            size="small"
                            disabled={!jsonConfig}
                        >
                            <ContentCopyIcon fontSize="small" />
                        </IconButton>
                    </Tooltip>
                </Box>
                <Box sx={{ mt: 1, fontStyle: 'italic', display: 'flex', alignItems: 'center' }}>
                    <LightbulbOutlineIcon sx={{ mr: 1 }} /> 
                    You can use the variables and classes above when writing CFEngine policy.
                </Box>
            </Dialog>
        </>
    );
};

export default ShowRenderedJson;
