\set ON_ERROR_STOP true
COMMENT ON DATABASE cfmp IS 'Mission Portal related settings not processed by API.';


DO $$
BEGIN
-- ci_session schema changed in version 3.12.0
-- look for session_id column in case current version is older than that
-- in which case we need to delete/re-create the table
IF EXISTS (SELECT * FROM information_schema.columns WHERE table_name = 'ci_sessions' AND column_name = 'session_id') THEN
  DROP TABLE IF EXISTS ci_sessions;
END IF;
END $$;

CREATE TABLE IF NOT EXISTS "ci_sessions" (
        "id" varchar(128) NOT NULL,
        "ip_address" varchar(45) NOT NULL,
        "timestamp" bigint DEFAULT 0 NOT NULL,
        "data" text DEFAULT '' NOT NULL
);

CREATE INDEX IF NOT EXISTS "ci_sessions_timestamp" ON "ci_sessions" ("timestamp");
COMMENT ON TABLE ci_sessions IS 'Information about current sessions.';
COMMENT ON COLUMN ci_sessions.id IS 'The unique identifier of the session, used as the primary key.';
COMMENT ON COLUMN ci_sessions.ip_address IS 'The IP address of the user who initiated the session.';
COMMENT ON COLUMN ci_sessions.timestamp IS 'The UNIX timestamp of the last activity of the session.';
COMMENT ON COLUMN ci_sessions.data IS 'The text data of the session, encoded in base64.';

-- Table: users

CREATE TABLE IF NOT EXISTS users
(
  id serial NOT NULL,
  username character varying(50),
  source character varying(20),
  last_login timestamp without time zone,
  remember_code character varying(50),
  dashboard integer DEFAULT NULL,
  CONSTRAINT id PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
COMMENT ON TABLE users IS 'User preferences and information about Mission Portal behavior.';
COMMENT ON COLUMN users.id IS 'The primary key of the user table.';
COMMENT ON COLUMN users.username IS 'The unique name of the user.';
COMMENT ON COLUMN users.source IS 'The source of the user account, such as internal or external (e.g. LDAP, Active Directory).';
COMMENT ON COLUMN users.last_login IS 'The timestamp of the last login of the user.';
COMMENT ON COLUMN users.remember_code IS 'The code used to remember the user login session.';
COMMENT ON COLUMN users.dashboard IS 'The id of the default dashboard for the user.';

ALTER TABLE users
ADD COLUMN IF NOT EXISTS seen_tour SMALLINT DEFAULT 0;
COMMENT ON COLUMN users.seen_tour IS 'A flag indicating whether the user has seen the tour of the Mission Portal.';

ALTER TABLE users
ADD COLUMN IF NOT EXISTS seen_wizard SMALLINT DEFAULT 0;
COMMENT ON COLUMN users.seen_wizard IS 'A flag indicating whether the user has seen the wizard of the Mission Portal.';

ALTER TABLE users
ADD COLUMN IF NOT EXISTS never_ask_timezone_change SMALLINT DEFAULT 0;
COMMENT ON COLUMN users.never_ask_timezone_change IS 'A flag indicating whether the user wants to be asked about changing the timezone.';

ALTER TABLE users
ADD COLUMN IF NOT EXISTS use_browser_time SMALLINT DEFAULT 0;
COMMENT ON COLUMN users.use_browser_time IS 'A flag indicating whether the user wants to use the browser time or the server time.';

ALTER TABLE users
ADD COLUMN IF NOT EXISTS dark_mode SMALLINT DEFAULT 0;
COMMENT ON COLUMN users.dark_mode IS 'A flag indicating whether the user prefers the dark mode or the light mode.';

ALTER TABLE users
ADD COLUMN IF NOT EXISTS pinned_items_version SMALLINT DEFAULT 0;
COMMENT ON COLUMN users.pinned_items_version IS 'This is used to add default pinned items which are added after this version.';

ALTER TABLE users
ADD COLUMN IF NOT EXISTS additional_data JSONB DEFAULT '{}';
COMMENT ON COLUMN users.additional_data IS 'A JSON object containing additional data about the user preferences and behavior.';

-- Table: astrolabeprofile

CREATE TABLE IF NOT EXISTS astrolabeprofile
(
  id serial NOT NULL,
  username character varying(50) NOT NULL,
  profileid character varying(50) NOT NULL,
  defaulttree boolean DEFAULT false,
  globaltree boolean DEFAULT false,
  sharedpermission character varying(50)[],
  sharedby character varying(50)[],
  data json,
  CONSTRAINT astrolabeprofile_pkey PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
COMMENT ON TABLE astrolabeprofile IS 'Information about Host trees such as who it was created by, who it is shared with and the definition of the host tree.';
COMMENT ON COLUMN astrolabeprofile.id IS 'The unique identifier of the profile, generated from a sequence.';
COMMENT ON COLUMN astrolabeprofile.username IS 'The username of the user who created or owns the profile.';
COMMENT ON COLUMN astrolabeprofile.profileid IS 'The name of the profile, such as OS, Services, etc.';
COMMENT ON COLUMN astrolabeprofile.defaulttree IS 'The flag that indicates whether the profile is the default one for the user or not.';
COMMENT ON COLUMN astrolabeprofile.globaltree IS 'The flag that indicates whether the profile is a global one for all users or not.';
COMMENT ON COLUMN astrolabeprofile.sharedpermission IS 'The array of usernames that the profile is shared with, may be empty.';
COMMENT ON COLUMN astrolabeprofile.sharedby IS 'The array of usernames that shared the profile with the user, may be empty.';
COMMENT ON COLUMN astrolabeprofile.data IS 'The JSON object that stores the profile data, such as label, classRegex, children, etc.';

-- Table: custom_search

-- 3.6.2 introduce report_uniq constraint
ALTER TABLE IF EXISTS report
  DROP CONSTRAINT  IF EXISTS   report_uniq;

ALTER TABLE IF EXISTS report 
  ADD CONSTRAINT report_uniq UNIQUE (username, label);


CREATE TABLE IF NOT EXISTS report
(
  id serial NOT NULL,
  username character varying(50) NOT NULL,
  url character varying(500),
  reportType character varying(50),
  reportCategory character varying(50),
  type character varying(50) NOT NULL,
  readonly INTEGER NOT NULL default 0,
  is_public INTEGER NOT NULL default 0,
  can_subscribe INTEGER NOT NULL default 0,
  is_subscribed INTEGER NOT NULL default 0,
  label character varying(500) NOT NULL,
  date timestamp without time zone,
  params text NOT NULL,
  sharedPermission character varying(50)[],
  sharedby character varying(50)[],
  AdvancedReportsData json,
  CONSTRAINT report_pkey PRIMARY KEY (id),
  CONSTRAINT report_uniq UNIQUE (username, label)
)
WITH (
  OIDS=FALSE
);
COMMENT ON TABLE report IS 'Information about saved reports.';
COMMENT ON COLUMN report.id IS 'The primary key of the report table.';
COMMENT ON COLUMN report.username IS 'The name of the user who saved the report.';
COMMENT ON COLUMN report.url IS 'The URL of the report.';
COMMENT ON COLUMN report.reportType IS 'The type of the report, such as compliance, inventory, or software update.';
COMMENT ON COLUMN report.reportCategory IS 'The category of the report, such as security, performance, or other.';
COMMENT ON COLUMN report.type IS 'The format of the report, such as pdf or csv.';
COMMENT ON COLUMN report.readonly IS 'A flag indicating whether the report is read-only or editable.';
COMMENT ON COLUMN report.is_public IS 'A flag indicating whether the report is public or private.';
COMMENT ON COLUMN report.can_subscribe IS 'A flag indicating whether the report can be subscribed to or not.';
COMMENT ON COLUMN report.is_subscribed IS 'A flag indicating whether the user is subscribed to the report or not.';
COMMENT ON COLUMN report.label IS 'The label of the report.';
COMMENT ON COLUMN report.date IS 'The date of the report.';
COMMENT ON COLUMN report.params IS 'The parameters of the report.';
COMMENT ON COLUMN report.sharedPermission IS 'A list of permissions that the report has been shared with.';
COMMENT ON COLUMN report.sharedby IS 'A list of users who have shared the report.';
COMMENT ON COLUMN report.AdvancedReportsData IS 'A JSON object containing the advanced reports data.';

ALTER TABLE report
ADD COLUMN IF NOT EXISTS export_id text DEFAULT NULL;
COMMENT ON COLUMN report.export_id IS 'The export id of the report, used for importing and exporting reports.';

ALTER TABLE report
ADD COLUMN IF NOT EXISTS meta_data JSONB DEFAULT '{}';
COMMENT ON COLUMN report.meta_data IS 'A JSON object containing the meta data of the report.';

ALTER TABLE report
DROP CONSTRAINT IF EXISTS export_uniq;

ALTER TABLE report
ADD CONSTRAINT export_uniq UNIQUE (username, export_id);

DO $$
BEGIN

IF EXISTS (SELECT 1
FROM information_schema.columns
WHERE  table_name='report_schedule' AND column_name='fromemail')
THEN
-- 3.6.1 renames fromemail TO emailfrom in report_schedule
ALTER TABLE IF EXISTS report_schedule RENAME COLUMN fromemail TO emailfrom;
END IF;

IF EXISTS (SELECT 1
FROM information_schema.columns
WHERE  table_name='report_schedule' AND column_name='toemail')
THEN
-- 3.6.1 renames fromemail TO emailfrom in report_schedule
ALTER TABLE IF EXISTS report_schedule RENAME COLUMN toemail TO emailfrom;
END IF;

IF EXISTS (SELECT 1
FROM information_schema.columns
WHERE  table_name='report_schedule' AND column_name='can_edit' AND column_name='can_delete' )
THEN
-- 3.6.1 removes can_edit and can_delete columns from report_schedule table
ALTER TABLE IF EXISTS report_schedule
  DROP COLUMN can_edit,
  DROP COLUMN can_delete;
END IF;
END$$;



CREATE TABLE IF NOT EXISTS report_schedule
(
  id character varying(500) NOT NULL,
  reportid int NOT NULL references report(id) ON DELETE CASCADE,
  userid character varying(50) NOT NULL,
  title character varying(500) NOT NULL,
  description character varying(500) NOT NULL,
  emailfrom character varying(500) NOT NULL,
  emailto character varying(500) NOT NULL,
  enabled INTEGER NOT NULL default 1,
  query text NOT NULL,
  outputtypes character varying(50)[],
  schedule character varying(500) NOT NULL,
  schedulehumanreadabletime character varying(500) NOT NULL,
  schedulename character varying(500) NOT NULL,
  site_url text NOT NULL,
  hostcontextsprofileid character varying(20),
  hostcontextspath text default NULL,
  hostcontexts json DEFAULT NULL,
  scheduledata json,
  CONSTRAINT report_schedule_pkey PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
COMMENT ON TABLE report_schedule IS 'Information about scheduled reports.';
COMMENT ON COLUMN report_schedule.id IS 'The unique identifier for the scheduled report.';
COMMENT ON COLUMN report_schedule.reportid IS 'The foreign key referencing the associated report.';
COMMENT ON COLUMN report_schedule.userid IS 'The user ID associated with the scheduled report.';
COMMENT ON COLUMN report_schedule.title IS 'The title of the scheduled report.';
COMMENT ON COLUMN report_schedule.description IS 'The description of the scheduled report.';
COMMENT ON COLUMN report_schedule.emailfrom IS 'The email address from which the report is sent.';
COMMENT ON COLUMN report_schedule.emailto IS 'The email address to which the report is sent.';
COMMENT ON COLUMN report_schedule.enabled IS 'Flag indicating whether the scheduled report is enabled.';
COMMENT ON COLUMN report_schedule.query IS 'The SQL query that defines the report for the scheduled task.';
COMMENT ON COLUMN report_schedule.outputtypes IS 'Array of output types for the scheduled report.';
COMMENT ON COLUMN report_schedule.schedule IS 'The schedule for running the report.';
COMMENT ON COLUMN report_schedule.schedulehumanreadabletime IS 'Human-readable representation of the schedule time.';
COMMENT ON COLUMN report_schedule.schedulename IS 'The name associated with the schedule.';
COMMENT ON COLUMN report_schedule.site_url IS 'The URL associated with the scheduled report.';
COMMENT ON COLUMN report_schedule.hostcontextsprofileid IS 'The profile ID associated with the host contexts.';
COMMENT ON COLUMN report_schedule.hostcontextspath IS 'The path associated with the host contexts.';
COMMENT ON COLUMN report_schedule.hostcontexts IS 'JSON data representing the subset of hosts that the report should be filtered for. If not defined the scheduled report includes all hosts the userid is allowed to see based on RBAC and data reported by the host.';
COMMENT ON COLUMN report_schedule.scheduledata IS 'JSON data containing details about the schedule.';

ALTER TABLE report_schedule
ADD COLUMN IF NOT EXISTS excludedhosts json DEFAULT NULL;
COMMENT ON COLUMN report_schedule.excludedhosts IS 'JSON data representing excluded hosts for the scheduled report.';

ALTER TABLE "report_schedule"
ADD COLUMN IF NOT EXISTS skipmailing boolean NULL DEFAULT 'false';
COMMENT ON COLUMN report_schedule.skipmailing IS 'Flag indicating whether mailing is skipped for the scheduled report.';

CREATE TABLE IF NOT EXISTS app
(
  displayindex integer,
  filepath text,
  hascontroller integer,
  icon character varying(50),
  meta json,
  showappslist integer,
  state integer,
  url text,
  id character varying(100),
  CONSTRAINT app_pk PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
COMMENT ON TABLE app IS 'Information about Mission Portal applications.';
COMMENT ON COLUMN app.displayindex IS 'The display order of the app in the Mission Portal menu.';
COMMENT ON COLUMN app.filepath IS 'The path of the app module in the application directory.';
COMMENT ON COLUMN app.hascontroller IS 'The flag that indicates whether the app has a controller file or not.';
COMMENT ON COLUMN app.icon IS 'The name of the app icon file in the images directory.';
COMMENT ON COLUMN app.meta IS 'The JSON object that stores the app metadata, such as name, description, license, etc.';
COMMENT ON COLUMN app.showappslist IS 'The flag that indicates whether the app is visible in the Mission Portal menu or not.';
COMMENT ON COLUMN app.state IS 'The state of the app, such as 1 or 0.';
COMMENT ON COLUMN app.url IS 'The URL of the app in the Mission Portal.';
COMMENT ON COLUMN app.id IS 'The unique identifier of the app, used as the primary key.';

ALTER TABLE "app"
DROP COLUMN IF EXISTS onlyforadmin;

ALTER TABLE "app"
ADD COLUMN IF NOT EXISTS rbac_id character varying(50) NULL;
COMMENT ON COLUMN app.rbac_id IS 'The identifier of the RBAC permission that the app requires, may be null.';



-- Table: eventslog

CREATE TABLE IF NOT EXISTS eventslog
(
  id serial NOT NULL,
  username character varying(100),
  item_id character varying(100), -- Alert id
  item_type character varying,
  item_name character varying(500),
  tags character varying(500)[],
  "time" timestamp without time zone,
  severity character varying(20),
  message text,
  status   character varying(10),
  CONSTRAINT eventslog_pkey PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
COMMENT ON TABLE eventslog IS 'Event logs.';
COMMENT ON COLUMN eventslog.id IS 'Unique identifier for the event log entry.';
COMMENT ON COLUMN eventslog.username IS 'Username associated with the event log entry.';
COMMENT ON COLUMN eventslog.item_id IS 'Identifier associated with the item triggering the event (e.g., alert id).';
COMMENT ON COLUMN eventslog.item_type IS 'Type of the item triggering the event (e.g., host, alerts).';
COMMENT ON COLUMN eventslog.item_name IS 'Name of the item triggering the event.';
COMMENT ON COLUMN eventslog.tags IS 'Tags associated with the event log entry.';
COMMENT ON COLUMN eventslog.time IS 'Timestamp when the event occurred.';
COMMENT ON COLUMN eventslog.severity IS 'Severity level of the event such as low, medium, high. Not all events specify a severity.';
COMMENT ON COLUMN eventslog.message IS 'Detailed message describing the event.';
COMMENT ON COLUMN eventslog.status IS 'Status of the event (e.g., triggered, cleared).';

-- 3.7.0 dashboard sharing
CREATE TABLE IF NOT EXISTS dashboard_dashboards
(
  id serial NOT NULL,
  name character varying(200),
  username character varying(20),
  public integer DEFAULT 0,
  widgets character varying(200),
  CONSTRAINT dashboard_dashboards_pkey PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
COMMENT ON TABLE dashboard_dashboards IS 'User dashboards and configuration.';
COMMENT ON COLUMN dashboard_dashboards.id IS 'The primary key of the dashboard table.';
COMMENT ON COLUMN dashboard_dashboards.name IS 'The name of the dashboard.';
COMMENT ON COLUMN dashboard_dashboards.username IS 'The name of the user who owns the dashboard.';
COMMENT ON COLUMN dashboard_dashboards.public IS 'A flag indicating whether the dashboard is public or private.';
COMMENT ON COLUMN dashboard_dashboards.widgets IS 'A comma separated list of widget ids that are displayed on the dashboard.';

ALTER TABLE IF EXISTS  dashboard_dashboards
ADD COLUMN IF NOT EXISTS  sharedWith jsonb NULL
DEFAULT '{"roles": [], "users": [], "sharedWithAll": true}';
COMMENT ON COLUMN dashboard_dashboards.sharedWith IS 'A JSON object containing the roles, users, and sharedWithAll flag that determine the sharing settings of the dashboard.';

-- 3.6.1 intruduced ordering column to dashboard_widgets
ALTER TABLE IF EXISTS dashboard_widgets
  ADD COLUMN IF NOT EXISTS ordering integer DEFAULT 0;


CREATE TABLE IF NOT EXISTS dashboard_widgets
(
  id serial NOT NULL,
  name character varying(500),
  type character varying(20),
  username character varying(50),
  ordering integer DEFAULT 0,
  dashboardid integer,
  CONSTRAINT dashboard_widgets_pkey PRIMARY KEY (id),
  CONSTRAINT dashboard_widgets_dashboardid_fkey FOREIGN KEY (dashboardid)
      REFERENCES dashboard_dashboards (id)  ON DELETE CASCADE
)
WITH (
  OIDS=FALSE
);
COMMENT ON TABLE dashboard_widgets IS 'User configurations for dashboard widgets.';
COMMENT ON COLUMN dashboard_widgets.id IS 'Unique identifier for the dashboard widget.';
COMMENT ON COLUMN dashboard_widgets.name IS 'Name of the dashboard widget.';
COMMENT ON COLUMN dashboard_widgets.type IS 'Type of the dashboard widget (e.g., inventory, alerts, hostCount).';
COMMENT ON COLUMN dashboard_widgets.username IS 'Username of the user who configured the dashboard widget.';
COMMENT ON COLUMN dashboard_widgets.ordering IS 'Ordering of the dashboard widget in the dashboard.';
COMMENT ON COLUMN dashboard_widgets.dashboardid IS 'Identifier for the associated dashboard.';

ALTER TABLE IF EXISTS "dashboard_widgets"
ADD COLUMN IF NOT EXISTS payload jsonb NULL DEFAULT '{}';

ALTER TABLE "dashboard_widgets" ALTER "payload" TYPE jsonb;
COMMENT ON COLUMN dashboard_widgets.payload IS 'JSON payload containing additional configuration for the dashboard widget.';

CREATE INDEX IF NOT EXISTS _ordering ON dashboard_widgets (ordering) WITH (FILLFACTOR = 90);

CREATE TABLE IF NOT EXISTS dashboard_rules
(
  id serial NOT NULL,
  name character varying(200),
  description character varying(1000),
  type character varying(20),
  username character varying(20),
  policyconditions json,
  inventoryconditions json,
  softwareupdateconditions json,
  CONSTRAINT dashboard_rules_pkey PRIMARY KEY (id),
  CONSTRAINT dashboard_rules_name_key UNIQUE (name,username)
)
WITH (
  OIDS=FALSE
);
COMMENT ON TABLE dashboard_rules IS 'User-defined dashboard alert rules.';
COMMENT ON COLUMN dashboard_rules.id IS 'Unique identifier for the dashboard rule.';
COMMENT ON COLUMN dashboard_rules.name IS 'Name of the dashboard rule.';
COMMENT ON COLUMN dashboard_rules.description IS 'Description of the dashboard rule.';
COMMENT ON COLUMN dashboard_rules.type IS 'Type of the dashboard rule (e.g., policy, softwareupdate, inventory).';
COMMENT ON COLUMN dashboard_rules.username IS 'Username of the user who created the dashboard rule.';
COMMENT ON COLUMN dashboard_rules.policyconditions IS 'Dashboard rule conditions for checks based on promise outcomes such as KEPT, NOT_KEPT, and REPAIRED. (JSON object)';
COMMENT ON COLUMN dashboard_rules.inventoryconditions IS 'JSON object of conditions for inventory-based dashboard rules.';
COMMENT ON COLUMN dashboard_rules.softwareupdateconditions IS 'JSON object of conditions for software update-based dashboard rules.';

-- 3.16.0 add category and severity columns
ALTER TABLE IF EXISTS dashboard_rules
  ADD COLUMN IF NOT EXISTS category text DEFAULT NULL;
COMMENT ON COLUMN dashboard_rules.category IS 'Category assigned to the dashboard rule.';

ALTER TABLE IF EXISTS dashboard_rules
  ADD COLUMN IF NOT EXISTS severity text DEFAULT NULL;
COMMENT ON COLUMN dashboard_rules.severity IS 'Severity level assigned to the dashboard rule such as low, medium, high.';

ALTER TABLE IF EXISTS dashboard_rules
  ADD COLUMN IF NOT EXISTS hostcontexts json DEFAULT NULL;
COMMENT ON COLUMN dashboard_rules.hostcontexts IS 'JSON object describing the set of hosts the limiting the hosts that should be considered when checking the rule. If not set the condition is checked for against all hosts the user has access to based on RBAC and host reported data.';

ALTER TABLE IF EXISTS dashboard_rules
  ADD COLUMN IF NOT EXISTS conditionMustBeMet boolean DEFAULT FALSE;
COMMENT ON COLUMN dashboard_rules.conditionMustBeMet IS 'Flag indicating whether conditions must be met for the dashboard rule.';

-- 3.12.1 drop sketchactivation
ALTER TABLE dashboard_rules DROP COLUMN IF EXISTS sketchactivation;

-- 3.12.0 add new alert custom condition
ALTER TABLE IF EXISTS dashboard_rules
  ADD COLUMN IF NOT EXISTS customconditions json DEFAULT '{}';
COMMENT ON COLUMN dashboard_rules.customconditions IS 'Custom dashboard conditions (for widgets), which use SQL queries returning hostkeys of affected hosts (JSON object)';

-- 3.15.0 add new alert file changed condition
ALTER TABLE IF EXISTS dashboard_rules
  ADD COLUMN IF NOT EXISTS filechangedconditions json DEFAULT '{}';
COMMENT ON COLUMN dashboard_rules.filechangedconditions IS 'File changed conditions for the dashboard rule.';

ALTER TABLE dashboard_rules
    ADD COLUMN IF NOT EXISTS export_id text DEFAULT NULL;
COMMENT ON COLUMN dashboard_rules.export_id IS 'Identifier for exporting dashboard rules.';

ALTER TABLE dashboard_rules
DROP CONSTRAINT IF EXISTS dashboard_rules_export_uniq;

ALTER TABLE dashboard_rules
    ADD CONSTRAINT dashboard_rules_export_uniq UNIQUE (username, export_id);

ALTER TABLE dashboard_rules
    ALTER "description" TYPE text,
    ALTER "name" TYPE text;

CREATE TABLE IF NOT EXISTS dashboard_alerts
(
  id serial NOT NULL,
  ruleid integer,
  failhosts integer DEFAULT 0,
  lastcheck integer,
  lasteventtime integer,
  laststatuschange integer,
  servertime integer,
  pause integer DEFAULT 0,
  paused integer DEFAULT 0,
  name character varying(500),
  severity character varying(10),
  site_url text,
  status character varying(32),
  totalhosts integer,
  username character varying(50),
  widgetname character varying(100),
  emailtonotify character varying(100),
  reminder integer,
  widgetid integer,
  hostcontextsprofileid character varying(20),
  hostContexts json DEFAULT NULL,
  hostContextsPath text default NULL,
  CONSTRAINT dashboard_alerts_pkey PRIMARY KEY (id),
  CONSTRAINT dashboard_alerts_ruleid_fkey FOREIGN KEY (ruleid)
      REFERENCES dashboard_rules (id)  ON DELETE CASCADE,
  CONSTRAINT dashboard_alerts_widgetid_fkey FOREIGN KEY (widgetid)
      REFERENCES dashboard_widgets (id)  ON DELETE CASCADE,
  CONSTRAINT dashboard_alerts_uniq UNIQUE (username, name, widgetid)
)
WITH (
  OIDS=FALSE
);
COMMENT ON TABLE dashboard_alerts IS 'User dashboards alerts status.';
COMMENT ON COLUMN dashboard_alerts.id IS 'The primary key of the dashboard alerts table.';
COMMENT ON COLUMN dashboard_alerts.ruleid IS 'The id of the rule that triggered the alert.';
COMMENT ON COLUMN dashboard_alerts.failhosts IS 'The number of hosts that failed the rule.';
COMMENT ON COLUMN dashboard_alerts.lastcheck IS 'The timestamp of the last check of the rule.';
COMMENT ON COLUMN dashboard_alerts.lasteventtime IS 'The timestamp of the last event that caused the alert status to change.';
COMMENT ON COLUMN dashboard_alerts.laststatuschange IS 'The timestamp of the last change of the alert status.';
COMMENT ON COLUMN dashboard_alerts.servertime IS 'The timestamp of the server time when the alert was generated.';
COMMENT ON COLUMN dashboard_alerts.pause IS 'The timestamp indicating when the alert was paused.';
COMMENT ON COLUMN dashboard_alerts.paused IS 'A flag indicating whether the alert was paused by the user or not.';
COMMENT ON COLUMN dashboard_alerts.name IS 'The name of the alert.';
COMMENT ON COLUMN dashboard_alerts.severity IS 'The severity level of the alert, such as high, medium, or low.';
COMMENT ON COLUMN dashboard_alerts.site_url IS 'The URL of the Mission Portal host where the alert is displayed.';
COMMENT ON COLUMN dashboard_alerts.status IS 'The status of the alert, such as success, fail, or warning.';
COMMENT ON COLUMN dashboard_alerts.totalhosts IS 'The total number of hosts that are affected by the rule.';
COMMENT ON COLUMN dashboard_alerts.username IS 'The name of the user who created the alert.';
COMMENT ON COLUMN dashboard_alerts.widgetname IS 'The name of the widget that shows the alert.';
COMMENT ON COLUMN dashboard_alerts.emailtonotify IS 'The email address of the user who will be notified of the alert.';
COMMENT ON COLUMN dashboard_alerts.reminder IS 'The frequency of the reminder email for the alert.';
COMMENT ON COLUMN dashboard_alerts.widgetid IS 'The id of the widget that shows the alert.';
COMMENT ON COLUMN dashboard_alerts.hostcontextsprofileid IS 'The id of the host contexts profile that defines the scope of the alert.';
COMMENT ON COLUMN dashboard_alerts.hostContexts IS 'A JSON object containing the host contexts that define the scope of the alert.';
COMMENT ON COLUMN dashboard_alerts.hostContextsPath IS 'The path of the host contexts that define the scope of the alert.';

ALTER TABLE dashboard_alerts
ADD COLUMN IF NOT EXISTS excludedhosts json DEFAULT NULL;
COMMENT ON COLUMN dashboard_alerts.excludedhosts IS 'A JSON object describing hosts that should be excluded from checking the alert.';

-- 3.6.2 introduce variables_dictionary_attribute_name constraint
ALTER TABLE IF EXISTS variables_dictionary
  DROP CONSTRAINT  IF EXISTS   variables_dictionary_attribute_name;

ALTER TABLE IF EXISTS variables_dictionary 
  ADD CONSTRAINT variables_dictionary_attribute_name UNIQUE (attribute_name);

CREATE TABLE IF NOT EXISTS variables_dictionary
(
  id serial NOT NULL,
  attribute_name character varying(200),
  category character varying(200),
  readonly integer default 0,
  type character varying(200),
  convert_function character varying(200) NULL,
  CONSTRAINT variables_dictionary_pkey PRIMARY KEY (id),
  CONSTRAINT variables_dictionary_attribute_name UNIQUE (attribute_name)
)
WITH (
  OIDS=FALSE
);
COMMENT ON TABLE variables_dictionary IS 'Information about reported inventory attributes.';
COMMENT ON COLUMN variables_dictionary.id IS 'The unique identifier for the variable in the dictionary.';
COMMENT ON COLUMN variables_dictionary.attribute_name IS 'The name of the attribute represented by the variable.';
COMMENT ON COLUMN variables_dictionary.category IS 'The category to which the attribute belongs.';
COMMENT ON COLUMN variables_dictionary.readonly IS 'Flag indicating whether the attribute is read-only.';
COMMENT ON COLUMN variables_dictionary.type IS 'The data type of the attribute such as string, slist, int, real.';
COMMENT ON COLUMN variables_dictionary.convert_function IS 'The conversion function applied to the attribute such as cf_clearslist (if any).';

-- Table: dashboard_scripts

CREATE TABLE IF NOT EXISTS dashboard_scripts
(
  id serial NOT NULL,
  name text,
  description text,
  script_name text,
  type text,
  CONSTRAINT unique_script_name UNIQUE (name),
  CONSTRAINT dashboard_scripts_pkey PRIMARY KEY (id)
)
WITH (
OIDS=FALSE
);
COMMENT ON TABLE dashboard_scripts IS 'Table containing scripts available for association with alerts.';
COMMENT ON COLUMN dashboard_scripts.id IS 'Unique identifier for the script entry.';
COMMENT ON COLUMN dashboard_scripts.name IS 'Name of the script.';
COMMENT ON COLUMN dashboard_scripts.description IS 'Description of the script.';
COMMENT ON COLUMN dashboard_scripts.script_name IS 'Name of the actual script file.';
COMMENT ON COLUMN dashboard_scripts.type IS 'Type of the script. (not used)';

CREATE TABLE IF NOT EXISTS dashboard_alerts_script
(
  alert_id integer NOT NULL,
  script_id integer,
  CONSTRAINT alert_id_fk FOREIGN KEY (alert_id)
  REFERENCES dashboard_alerts (id) MATCH SIMPLE
  ON UPDATE NO ACTION ON DELETE CASCADE,
  CONSTRAINT script_id_fk FOREIGN KEY (script_id)
  REFERENCES dashboard_scripts (id) MATCH SIMPLE
  ON UPDATE NO ACTION ON DELETE CASCADE
)
WITH (
OIDS=FALSE
);
COMMENT ON TABLE dashboard_alerts_script IS 'Association of script with dashboard alert.';
COMMENT ON COLUMN dashboard_alerts_script.alert_id IS 'The id of the dashboard alert that is associated with a script.';
COMMENT ON COLUMN dashboard_alerts_script.script_id IS 'The id of the script that is associated with a dashboard alert.';

-- Index: fki_alert_id_fk
CREATE INDEX IF NOT EXISTS fki_alert_id_fk
ON dashboard_alerts_script
USING btree
(alert_id);

-- Index: fki_script_id_fk
CREATE INDEX IF NOT EXISTS fki_script_id_fk
ON dashboard_alerts_script
USING btree
(script_id);

CREATE TABLE IF NOT EXISTS mail_settings (
    key character varying PRIMARY KEY,
    value text
);
COMMENT ON TABLE mail_settings IS 'Global email settings.';
COMMENT ON COLUMN mail_settings.key IS 'Key representing a specific email setting.';
COMMENT ON COLUMN mail_settings.value IS 'Value associated with the email setting key.';

CREATE TABLE  IF NOT EXISTS customization (
   key character varying PRIMARY KEY,
   value text
);
COMMENT ON TABLE customization IS 'Stores Mission Portal UI customization config.';
COMMENT ON COLUMN customization.key IS 'The name of the customization option such as logo_on_login, login_text, header_color, etc.';
COMMENT ON COLUMN customization.value IS 'The value of the customization option.';

CREATE TABLE IF NOT EXISTS compliance_score (
  report_id integer NOT NULL,
  username text NOT NULL,
  score integer NOT NULL,
  update_ts timestamptz NOT NULL,
  fail_checks integer NOT NULL,
  total integer NOT NULL,
  CONSTRAINT compliance_score_uniq UNIQUE (username, report_id)
);
COMMENT ON TABLE compliance_score IS 'Compliance reports score.';
COMMENT ON COLUMN compliance_score.report_id IS 'The id of the compliance report that the user has generated.';
COMMENT ON COLUMN compliance_score.username IS 'The name of the user who has generated the compliance report.';
COMMENT ON COLUMN compliance_score.score IS 'The percentage of compliance checks that the user has passed.';
COMMENT ON COLUMN compliance_score.update_ts IS 'The timestamp of the last update of the compliance report.';
COMMENT ON COLUMN compliance_score.fail_checks IS 'The number of compliance checks that the user has failed.';
COMMENT ON COLUMN compliance_score.total IS 'The total number of compliance checks that the user has performed.';

DO $$
BEGIN
IF NOT EXISTS (SELECT 1 FROM pg_type WHERE typname = 'pinned_type') THEN
  CREATE TYPE pinned_type AS ENUM ('inventory', 'class', 'variable');
END IF;
END$$;

CREATE TABLE IF NOT EXISTS pinned_items (
  "id" bigserial PRIMARY KEY,
  "username" text NOT NULL,
  "type" pinned_type NOT NULL,
  "name" text NOT NULL,
  "created_at" timestamptz NOT NULL DEFAULT now()
);
COMMENT ON TABLE pinned_items IS 'Pinned inventory, class, or variable items.';
COMMENT ON COLUMN pinned_items.id IS 'Unique identifier for the pinned item.';
COMMENT ON COLUMN pinned_items.username IS 'Username of the user who pinned the item.';
COMMENT ON COLUMN pinned_items.type IS 'Type of the pinned item (e.g., inventory, class, variable).';
COMMENT ON COLUMN pinned_items.name IS 'Name of the pinned item.';
COMMENT ON COLUMN pinned_items.created_at IS 'Timestamp indicating when the item was pinned.';

ALTER TABLE IF EXISTS pinned_items
  DROP CONSTRAINT IF EXISTS pinned_items_PK;

ALTER TABLE IF EXISTS pinned_items
  ADD CONSTRAINT pinned_items_PK UNIQUE ("username", "type", "name");

CREATE TABLE IF NOT EXISTS favourite_reports (
  "report_id" BIGINT NOT NULL REFERENCES public.report ON DELETE CASCADE,
  "username" TEXT NOT NULL,
  "created_at" timestamptz NOT NULL DEFAULT now()
);
ALTER TABLE IF EXISTS favourite_reports DROP CONSTRAINT IF EXISTS favourite_reports_unique;
ALTER TABLE IF EXISTS favourite_reports ADD CONSTRAINT favourite_reports_unique UNIQUE ("username", "report_id");
COMMENT ON TABLE favourite_reports IS 'Table associating favorited reports with users.';
COMMENT ON COLUMN favourite_reports.report_id IS 'Identifier of the favorite report.';
COMMENT ON COLUMN favourite_reports.username IS 'Username of the user who marked the report as a favorite.';
COMMENT ON COLUMN favourite_reports.created_at IS 'Timestamp indicating when the report was marked as a favorite.';

CREATE OR REPLACE FUNCTION insert_default_pin_items(uname TEXT) RETURNS VOID AS
$BODY$
BEGIN

INSERT INTO "pinned_items" ("username", "type", "name") VALUES
(uname,	'inventory',	'Policy Servers'),
(uname,	'inventory',	'Ports listening'),
(uname,	'inventory',	'Kernel Release'),
(uname,	'inventory',	'Interfaces'),
(uname,	'inventory',	'MAC addresses'),
(uname,	'class',	'debian'),
(uname,	'class',	'systemd'),
(uname,	'class',	'xen'),
(uname,	'class',	'redhat'),
(uname,	'variable',	'default.sys.uqhost'),
(uname,	'variable',	'default.sys.fqhost')
ON CONFLICT DO NOTHING;

RETURN;
END;
$BODY$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION pin_items_to_new_users()
  RETURNS trigger AS
$$
BEGIN
PERFORM insert_default_pin_items(NEW.username);
RETURN NULL;
END;
$$
LANGUAGE 'plpgsql';

DROP TRIGGER IF EXISTS user_insert_trigger on users;
CREATE TRIGGER user_insert_trigger
  AFTER INSERT ON users
  FOR EACH ROW EXECUTE PROCEDURE pin_items_to_new_users();


CREATE OR REPLACE FUNCTION pin_items_to_existing_users() RETURNS VOID AS
$BODY$
DECLARE
    _username TEXT;
BEGIN

FOR _username IN
SELECT username  FROM users WHERE pinned_items_version < 1
LOOP

PERFORM insert_default_pin_items(_username);

-- when we add more default pinned items we need to increment pinned_items_version --
UPDATE users SET pinned_items_version = 1 WHERE username = _username;

END LOOP;
RETURN;
END;
$BODY$ LANGUAGE plpgsql;

SELECT pin_items_to_existing_users();

CREATE OR REPLACE FUNCTION migrate_duplicate_reports() RETURNS VOID AS
$BODY$
DECLARE
_report RECORD;
BEGIN

FOR _report IN
SELECT label, params, advancedreportsdata::text, type, readonly, is_public, can_subscribe, is_subscribed, reportcategory
FROM report
WHERE username LIKE 'admin' OR username LIKE 'CFE_ROBOT'
GROUP BY label, params, advancedreportsdata::text, type, readonly, is_public, can_subscribe, is_subscribed, reportcategory
HAVING count(*) > 1

LOOP
DELETE FROM report WHERE label = _report.label AND username LIKE 'CFE_ROBOT';
UPDATE report SET username = 'CFE_ROBOT', meta_data = '{"source": "from CFEngine team", "migrated": true}'
WHERE label = _report.label AND username LIKE 'admin';
RAISE LOG 'OOTB Report "%" owner was changed from admin to CFE_ROBOT', _report.label;
END LOOP;

RETURN;
END;
$BODY$ LANGUAGE plpgsql;

COMMENT ON FUNCTION migrate_duplicate_reports IS 'Changes owner of previously created out-of-the-box reports from admin to CFE_ROBOT system user.';

GRANT ALL PRIVILEGES ON DATABASE cfmp TO cfmppostgres;
GRANT ALL PRIVILEGES ON ALL TABLES IN SCHEMA public TO cfmppostgres;
GRANT ALL PRIVILEGES ON ALL SEQUENCES IN SCHEMA public TO cfmppostgres;
