import React, { useEffect, useState, useCallback } from 'react';
import { makeStyles } from 'tss-react/mui';
import {
  TextField,
  Select,
  MenuItem,
  Checkbox,
  FormControlLabel,
  Divider,
  Typography,
  Box,
  FormHelperText,
  Button,
  OutlinedInput
} from '@mui/material';
import { ExpandMore as ExpandMoreIcon } from '@mui/icons-material';

import { useDispatch, useSelector } from 'react-redux';
import {
  getConfig,
  deleteConfig,
  createConfig,
  updateConfig,
  getPreferences,
  updatePreferences
} from './store/configSlice/thunks';

import { LLMConfig } from './store/configSlice';
import Spinner from '../common/Spinner';
import ComfirmationDialog from '../common/ComfirmationDialog';

const useStyles = makeStyles()(theme => ({
  formSection: {
    marginBottom: theme.spacing(3),
    maxWidth: 460,
    '& .MuiInputBase-input ': {
      marginBottom: 0,
      height: 32
    },
    '& .MuiSelect-root.MuiOutlinedInput-root': {
      width: 172,
      height: 40
    },
    '& .MuiSelect-icon': {
      right: 8
    }
  },
  fieldGroup: {
    display: 'flex',
    flexDirection: 'column',
    gap: theme.spacing(2.5)
  },
  divider: {
    margin: theme.spacing(4, 0)
  },
  checkboxSection: {
    display: 'flex',
    flexDirection: 'column',
    '& .MuiCheckbox-root:hover': {
      background: 'none'
    },
  },
  sectionTitle: {
    marginBottom: theme.spacing(1),
    fontWeight: 600
  },
  formRow: {
    display: 'flex',
    alignItems: 'center'
  },
  label: {
    width: 100,
    flexShrink: 0,
    marginRight: theme.spacing(3),
    color: theme.palette.text.primary,
    fontWeight: 500,
  },
  fieldContainer: {
    flex: 1,
  },
}));

const providers = [
  { value: 'openai', label: 'OpenAI' },
  { value: 'gemini', label: 'Gemini' },
  { value: 'mistral', label: 'Mistral' },
  { value: 'ollama', label: 'Ollama' },
  { value: 'anthropic', label: 'Anthropic' },
  { value: 'openai_like', label: 'OpenAI-like' }
] as const;

export const LlmSettings = () => {
  const { classes } = useStyles();
  const dispatch = useDispatch();
  const config = useSelector(state => state.llm.config);
  const preferencesState = useSelector(state => state.llm.preferences);
  const loading = useSelector(state => state.llm.loading);
  const [deleteDialogOpen, setDeleteDialogOpen] = useState<boolean>(false);
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [isSaving, setIsSaving] = useState<boolean>(false);
  const handleDeleteClick = () => setDeleteDialogOpen(true);
  const handleDeleteDialogClose = () => setDeleteDialogOpen(false);
  const [settings, setSettings] = useState<LLMConfig>({
    provider: '',
    model: '',
    token: '',
    base_url: ''
  });

  useEffect(() => {
    dispatch(getConfig());
    dispatch(getPreferences());
  }, []);

  useEffect(() => {
    if (!config) return;
    setSettings(prev => ({
      ...prev,
      provider: config.provider,
      model: config.model,
      token: config.token || '',
      base_url: config.base_url
    }));
  }, [config]);

  useEffect(() => {
    if (!preferencesState) return;

    setPreferences({ ...preferences, ...preferencesState });
  }, [preferencesState]);

  const [preferences, setPreferences] = useState({
    allowLlmViewAccessToAttributesNames: false,
    disableAI: false
  });

  const validateAllFields = useCallback(() => {
    const newErrors = {};

    if (!settings.provider) {
      newErrors.provider = 'Provider is required';
    }

    if (!settings.model && settings.provider) {
      newErrors.model = 'Model is required';
    }

    if (!settings.token && settings.provider && settings.provider !== 'ollama') {
      newErrors.token = 'Token is required';
    }

    if (settings.provider === 'ollama' || settings.provider === 'openai_like') {
      if (!settings.base_url) {
        newErrors.base_url = 'Base URL is required';
      } else {
        try {
          new URL(settings.base_url);
        } catch (error) {
          newErrors.base_url = 'Please enter a valid URL (e.g., http://localhost:11434/api/)';
        }
      }
    }


    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  }, [settings]);

  const handleInputChange = (field: keyof LLMConfig, value: string) => {
    setSettings(prev => ({ ...prev, [field]: value }));
    if (errors[field]) {
      setErrors(prev => ({ ...prev, [field]: '' }));
    }
  };

  const handlePreferencesChange = (field, value) => {
    dispatch(updatePreferences({ [field]: value }));
  };

  const handleDelete = async () => {
    await dispatch(deleteConfig(config.id));
    setSettings({ ...settings, provider: '', model: '', token: '', base_url: '', id: null });
    setDeleteDialogOpen(false);
  }

  const handleSave = async () => {
    setIsSaving(true);

    if (validateAllFields()) {
      const action = config?.id ? updateConfig({ id: config?.id, config: settings }) : createConfig(settings);
      await dispatch(action);
      dispatch(getConfig());
    }

    setIsSaving(false);
  };

  const shouldShowField = (field) => {
    switch (field) {
      case 'token':
        return settings.provider && settings.provider !== 'ollama';
      case 'base_url':
        return settings.provider === 'ollama' || settings.provider === 'openai_like';
      case 'model':
        return settings.provider;
      default:
        return true;
    }
  };

  return (
    <Box className="relative">
      {loading && <Spinner />}
      <Box className="pageTitle">
        <Typography component="h1">
          AI settings
        </Typography>
        {!loading && !config?.id && <Box sx={{ mt: 1 }}>Configure LLM to enable AI features for this organization.</Box>}
      </Box>
      <Box sx={{ mt: 3 }} className={classes.formSection}>
        <Box className={classes.fieldGroup}>
          <Box className={classes.formRow}>
            <Typography variant="body1" className={classes.label}>
              LLM provider:
            </Typography>
            <Box>
              <Select
                displayEmpty
                value={settings.provider}
                fullWidth
                onChange={(e) => handleInputChange('provider', e.target.value)}
                IconComponent={ExpandMoreIcon}
                input={<OutlinedInput />}
              >
                <MenuItem disabled value="">
                  Select a provider
                </MenuItem>
                {providers.map((provider) => (
                  <MenuItem key={provider.value} value={provider.value}>
                    {provider.label}
                  </MenuItem>
                ))}
              </Select>
              {errors.provider && <FormHelperText error>{errors.provider}</FormHelperText>}
            </Box>
          </Box>
          {shouldShowField('model') && (
            <Box className={classes.formRow}>
              <Typography variant="body1" className={classes.label}>
                Model:
              </Typography>
              <Box className={classes.fieldContainer}>
                <TextField
                  fullWidth
                  variant="outlined"
                  placeholder="gpt-4, claude-3-sonnet, mistral-large"
                  value={settings.model}
                  onChange={(e) => handleInputChange('model', e.target.value)}
                  error={!!errors.model}
                  helperText={errors.model}
                />
              </Box>
            </Box>
          )}
          {shouldShowField('token') && (
            <Box className={classes.formRow}>
              <Typography variant="body1" className={classes.label}>
                API token:
              </Typography>
              <Box className={classes.fieldContainer}>
                <TextField
                  fullWidth
                  variant="outlined"
                  type="password"
                  placeholder={config?.isTokenSet ? "Token is set" : "Enter your API token"}
                  value={settings.token}
                  onChange={(e) => handleInputChange('token', e.target.value)}
                  error={!!errors.token}
                  helperText={errors.token}
                />
              </Box>
            </Box>
          )}
          {shouldShowField('base_url') && (
            <Box className={classes.formRow}>
              <Typography variant="body1" className={classes.label}>
                Base URL:
              </Typography>
              <Box className={classes.fieldContainer}>
                <TextField
                  fullWidth
                  variant="outlined"
                  placeholder="http://localhost:11434/api/, https://api.example.com/v1"
                  value={settings.base_url}
                  onChange={(e) => handleInputChange('base_url', e.target.value)}
                  error={!!errors.base_url}
                  helperText={errors.base_url}
                />
              </Box>
            </Box>
          )}
        </Box>
      </Box>
      <Box className="flex flex-end">
        <Button
          color="primary"
          variant="contained"
          onClick={handleSave}
          disabled={isSaving}>
          {config?.id ? 'Update' : 'Save'}
        </Button>
        {config?.id &&
          <Button
            sx={{ ml: 2 }}
            variant="outlined"
            color="danger"
            onClick={handleDeleteClick}>
            Delete configuration
          </Button>
        }
      </Box>
      <Divider className={classes.divider} />
      <Box className={classes.checkboxSection}>
        <Typography variant="h6" className={classes.sectionTitle} color="textPrimary">
          Access permissions
        </Typography>
        <FormControlLabel
          sx={{ alignItems: 'flex-end' }}
          control={
            <Checkbox
              checked={preferences.allowLlmViewAccessToAttributesNames}
              onChange={(e) => handlePreferencesChange('allowLlmViewAccessToAttributesNames', e.target.checked)}
              color="primary"
            />
          }
          label={
            <Box sx={{ height: 32 }}>
              <Typography component="span">
                Allow access to the names of inventory variables defined in the system
              </Typography>
              <Typography variant="caption" color="text.secondary" display="block">
                Sharing attribute names helps the AI generate more accurate SQL queries by referencing the correct field names.
              </Typography>
            </Box>
          }
        />
        <FormControlLabel
          sx={{ mt: 1 }}
          control={
            <Checkbox
              checked={preferences.disableAI}
              onChange={(e) => handlePreferencesChange('disableAI', e.target.checked)}
              color="primary"
            />
          }
          label="Disable AI features and buttons"
        />
      </Box>
      <ComfirmationDialog
        confirmColor="error"
        open={deleteDialogOpen}
        onClose={handleDeleteDialogClose}
        onConfirm={handleDelete}
        title="Delete LLM configurations"
        message="Are you sure you want to delete the LLM configurations?"
        confirmText="Delete"
        cancelText="Cancel"
        size="xs"
      />
    </Box>
  );
};


export default LlmSettings;
