<?php

class Compliance_export extends Cf_Model
{
    private $totalHosts = 0;

    public function __construct()
    {
        parent::__construct();

        $this->load->model('dashboard/rules_model');
        $this->load->model('host_model');
    }

    public function getReportData(CF_Search $report, $username, $includes = [], $excludes = [])
    {
        $this->setTotalHostCount($username, $includes, $excludes);
        $reportAdvancedData = json_decode($report->getAdvancedReportsData()['SQL'], JSON_OBJECT_AS_ARRAY);
        $reportConditions = isset($reportAdvancedData['conditions']) ? $reportAdvancedData['conditions'] : [];

        $result = [];

        foreach ($reportConditions as $condition) {
            $result['savedConditions'][] = $this->getConditionStatistics($condition, $username, $includes, $excludes);
        }

        $result['totalHosts'] = $this->totalHosts;

        return $result;
    }

    private function setTotalHostCount($username, $includes, $excludes)
    {
        $queryResult = $this->advancedreports_model->runQuery(
            $username,
            'SELECT COUNT(*) FROM hosts',
            '',
            '',
            0,
            1,
            $includes,
            $excludes,
        );
        $this->totalHosts = (int) $queryResult['rows'][0][0];
    }

    private function getConditionStatistics($id, $username, $includes, $excludes)
    {
        [$condition] = $this->rules_model->get_rules_with_ids([$id], [], ['severity' => 'asc']);
        $params = $this->rules_model->getSQL($condition);

        // if filter exists then we need to add inventory CTE
        if (sizeof($params['filter']) > 0) {
            $this->addCteToInventoryQuery($params);
        }

        if (isset($condition->hostcontexts['includes'])) {
            $includes = array_merge($includes, $condition->hostcontexts['includes']);
        }

        if ($condition->getConditionMustBeMet() == true) {
            $sql = "SELECT COUNT(*) FROM hosts WHERE hostkey NOT IN ({$params['sql']})";
            $result = $this->advancedreports_model->runQuery($username, $sql, '', '', 0, 0, $includes, $excludes);
            $failHosts = intval($result['rows'][0][0]);
        } else {
            $result = $this->advancedreports_model->runQuery(
                $username,
                $params['sql'],
                '',
                '',
                0,
                0,
                $includes,
                $excludes,
            );
            $failHosts = sizeof($result['rows']);
        }

        $result = (array) $condition;

        $result['total'] =
            sizeof($includes) > 0 || sizeof($excludes) > 0
                ? (int) $this->host_model->getHostCountByContext($username, $includes, $excludes)
                : $this->totalHosts;

        $result['failed'] = $failHosts;
        $result['pass'] = $result['total'] - $result['failed'];
        $result['score'] = $result['failed'] == 0 ? 100 : round(($result['pass'] / $result['total']) * 100);

        return $result;
    }

    private function addCteToInventoryQuery(&$data)
    {
        $apiDispatchPath = FCPATH . 'api/dispatch.php';

        if (!file_exists($apiDispatchPath)) {
            throw new Exception(
                'Compliance_export::addCteToInventoryQueryWrong: path to API dispatch.php: ' . $apiDispatchPath,
            );
        }

        include_once $apiDispatchPath;

        $inventoryLib = new InventoryLib(new VariablesDictionaryModel());
        $where = $inventoryLib->applyFilter((object) $data);
        $cte = "WITH inventory_cte as (SELECT hostkey as filtered_hostkey FROM inventory_new $where) ";

        $re = '/^(\s+)*WITH/';
        preg_match($re, $data['sql'], $matches);

        // if user query already contain CTE then add inventory together with existing one
        if (sizeof($matches) > 0) {
            $data['sql'] = preg_replace($re, $cte . ', ', $data['sql']);
        } else {
            $data['sql'] = $cte . $data['sql'];
        }
    }

    public function generateCSV($reportData, $path)
    {
        $data = [['Name', 'Severity level', 'Compliance score', 'Category', 'Passing', 'All']];

        foreach ($reportData['savedConditions'] as $condition) {
            $data[] = [
                $condition['name'],
                $condition['severity'],
                $condition['score'],
                $condition['category'],
                $condition['pass'],
                $condition['total'],
            ];
        }

        $fp = fopen($path, 'w');

        foreach ($data as $fields) {
            fputcsv($fp, $fields);
        }

        fclose($fp);
    }

    public function generatePDF($reportData, $filename, $reportTitle, $reportDescription)
    {
        $this->load->model('PdfComplianceReports_model');
        $reportData['savedConditions'] = $this->groupConditionsByCategory($reportData['savedConditions']);
        $this->PdfComplianceReports_model->generateReport([$filename, $reportData, $reportTitle, $reportDescription]);
    }

    private function groupConditionsByCategory($conditions)
    {
        $result = [];
        foreach ($conditions as $condition) {
            if (!isset($result[$condition['category']])) {
                $result[$condition['category']] = [];
            }
            $result[$condition['category']][] = $condition;
        }
        return $result;
    }
}
