'use strict';

const groupsListController = BaseController.extend({
    init: function ($scope, $modal, $cookies, groupsService, paginationService, sharedGroupsService, personalGroupsService, inventoryHelper, globalHelper, $location, $rootScope) {
        this._super($scope);
        this.$modal = $modal;
        this.$cookies = $cookies;
        this.$location = $location;
        this.$scope = $scope;
        this.$rootScope = $rootScope;
        this.groupsService = groupsService;
        this.sharedGroupsService = sharedGroupsService;
        this.personalGroupsService = personalGroupsService;
        this.inventoryHelper = inventoryHelper;
        this.$scope.paginator = paginationService;
        this._globalHelper = globalHelper;
        this.$scope.paginator.paginationReset();
        let rowsPerPage = window.localStorage.getItem('groups.rowsPerPage');
        this.$scope.paginator.rowsPerPage = rowsPerPage ? rowsPerPage : this.$scope.paginator._setDefaultRowsPerPage();
        this.getGroups();

        this.$scope.duplicateGroup = this.duplicateGroup.bind(this);
        this.$scope.formatTime = common.time.format;
        this.$scope.sortReorder = this.sortReorder.bind(this);
        this.$scope.remove = this.remove.bind(this);
        this.$scope.addFavorite = this.addFavorite.bind(this);
        this.$scope.removeFavorite = this.removeFavorite.bind(this);
        this.$scope.addGroup = this.addGroup.bind(this);
        this.$scope.isShared = this.isShared;
        this.$scope.isPersonal = this.isPersonal;
        this.$scope.getUserName = this.getUserName.bind(this);
        this.$scope.favoriteGroups = [];

        let self = this;
        this.inventoryHelper.getInventoryVariables('','','').then(
            function(result) {
                if (!result.CFE_mapping || Object.keys(result.CFE_mapping).length == 0) {
                    self.$scope.no_inventory = true;
                } else {
                    self.$scope.have_inventory = true;
                }
            },
            function(error) {
                self.$scope.no_inventory = true;
            }
        );
    },

    defineListeners: function () {
        let self = this;

        self.$scope.$on('paginatorChange', function () {
            window.localStorage.setItem('groups.rowsPerPage', self.$scope.paginator.rowsPerPage);
            self.getGroups();
        });

        self.$scope.$watch('type', type => self.isShared(self.$scope.type) && (this.$rootScope.hasRedirectedToShared = true));
    },

    getGroups: function () {
        let self = this;
        const params = $.param({
            skip: (self.$scope.paginator.page - 1) * self.$scope.paginator.rowsPerPage,
            limit: self.$scope.paginator.rowsPerPage,
            sortColumn: this.$scope.sort.column,
            sortDescending: this.$scope.sort.descending ? 'true' : 'false',
            includeHostsCount: 'yes'
        });

        self.groupsService.list(params).then(response => {
            self.$scope.groups = response.data.data;
            // redirect to shared groups list if:
            // groups list is personal and no groups found and redirect did not happen before
            if (self.isPersonal(self.$scope.type) && response.data.meta.total === 0 && !this.$rootScope.hasRedirectedToShared) {
                self.$location.path('/shared');
            }
            self.$scope.paginator.refreshPaginator(response.data.meta.total);
        });
    },

    isShared: type => type == 'shared',

    isPersonal: type => type === 'personal',

    duplicateGroup: function (id, type, isShared) {
        let self = this;
        this.modalInstance = this.$modal.open({
            templateUrl: 'duplicateGroup.html',
            backdrop: 'static',
            keyboard: true,
            windowClass: 'duplicate-group-dialog',
            controller: function ($scope, $modalInstance, $parent, group) {
                let self = this;

                $scope.group = group;
                $scope.data = {type};
                $scope.close = function () {
                    $modalInstance.close('cancel');
                };

                const currentGroupService = $parent.getGroupsService.call($parent, isShared);
                $scope.$watch('data.type', type => {
                    this.newGroupService = $parent.isShared(type) ? $parent.sharedGroupsService : $parent.personalGroupsService;
                });

                $scope.duplicate = async function () {
                    try {
                        common.globalSpinner.show();
                        let group = (await currentGroupService.get(id)).data;
                        group.name += ' duplicate';
                        if ($parent.isShared($scope.data.type)) {
                            const groupWithSameNameResponse = (
                                await self.newGroupService.list(`searchQuery=${group.name}&sortColumn=name&sortDescending=true&limit=1`)
                            ).data.data;
                            if (groupWithSameNameResponse.length > 0) {
                                const groupWithSameName = groupWithSameNameResponse[0];
                                const lastNumberInName = groupWithSameName.name.match(/\d+$/);
                                group.name += ' ' + (lastNumberInName ? (parseInt(lastNumberInName[0]) + 1) : 1).toString();
                            }
                        }
                        const newGroupId = (await self.newGroupService.create(group)).data.id;
                        document.location.href = `/groups/${$scope.data.type}/${newGroupId}`;
                    } catch (exception) {
                        notify.error(exception.data);
                        common.globalSpinner.hide();
                    }
                };
            },
            resolve: {
                $parent: function () {
                    return self;
                },
                group: function () {
                    return {id, type};
                }
            }
        });
    },

    sortReorder: function (column, descending) {
        this.$scope.sort.column = column;
        this.$scope.sort.descending = descending;
        this.getGroups();
    },

    getGroupsService: function (isShared) {
        return isShared ? this.sharedGroupsService : this.personalGroupsService;
    },

    remove: function (id, name, isShared = false) {
        let self = this;
        this.modalInstance = this.$modal.open({
            templateUrl: 'removeGroup.html',
            backdrop: 'static',
            keyboard: true,
            windowClass: 'remove-group-dialog',
            controller: function ($scope, $modalInstance, $parent, group) {
                $scope.group = group;
                $scope.isShared = isShared;
                $scope.close = function () {
                    $modalInstance.close('cancel');
                };

                const groupService = $parent.getGroupsService.call($parent, isShared);
                $scope.remove = function () {
                    groupService.remove($scope.group.id).then(
                        () => {
                            notify.success('Group successfully removed.');
                            $scope.close();
                            $parent.getGroups();
                        },
                        error => {
                            notify.error(error.data);
                        }
                    );
                };
            },
            resolve: {
                $parent: function () {
                    return self;
                },
                group: function () {
                    return {id, name};
                }
            }
        });
    },

    addGroup: function () {
        this.groupsService.addGroup().then(
            response => document.location.href = `/groups/${this.$scope.type}/${response.data.id}`,
            error => notify.error(error.data)
        )
    },

    addFavorite: function (id, type) {
        const groupService = this.getGroupsService(this.isShared(type));
        groupService.addToFavorites(id).then(
            () => this.changeFavoriteFlag(id, true)
        )
    },

    removeFavorite: function (id, type) {
        const groupService = this.getGroupsService(this.isShared(type));
        groupService.removeFromFavorites(id).then(
            () => this.changeFavoriteFlag(id, false)
        )
    },

    changeFavoriteFlag: function (id, value) {
        const key = Object.keys(this.$scope.groups).find(key => this.$scope.groups[key].id === id);
        if (this.$scope.groups.hasOwnProperty(key)) {
            this.$scope.groups[key].is_favorite = value;
        }
    },
    
    getUserName: function (group) {
        return this._globalHelper.getUserName(group?.meta_data?.source, group.owner || group.creator);
    }
});

const personalGroupsListController = groupsListController.extend({
    init: function ($scope, $routeParams, $modal, $cookies, personalGroupsService, paginationService, sharedGroupsService, inventoryHelper, globalHelper, $location, $rootScope) {
        $scope.sort = {
            column: 'name',
            descending: false
        }
        this.inventoryHelper = inventoryHelper;
        this._super($scope, $modal, $cookies, personalGroupsService, paginationService, sharedGroupsService, personalGroupsService, inventoryHelper, globalHelper, $location, $rootScope);
        this.$scope.type = 'personal';
        $cookies.put('grouplist', 'personal', {'path':'/'});
    },

    defineListeners: function () {
        this._super()
        let self = this;
    }

});

personalGroupsListController.$inject = ['$scope', '$routeParams', '$modal', '$cookies', 'personalGroupsService', 'IEPaginationService', 'sharedGroupsService', 'inventoryHelper', 'globalHelper', '$location', '$rootScope'];

const sharedGroupsListController = groupsListController.extend({
    init: function ($scope, $routeParams, $modal, $cookies, sharedGroupsService, paginationService, personalGroupsService, inventoryHelper, globalHelper, $location, $rootScope) {
        $scope.sort = {
            column: 'priority',
            descending: false
        }
        this._super($scope, $modal, $cookies, sharedGroupsService, paginationService, sharedGroupsService, personalGroupsService, inventoryHelper, globalHelper, $location, $rootScope);
        this.sharedGroupsService = sharedGroupsService;
        this.inventoryHelper = inventoryHelper;
        this.$scope.type = 'shared';
        $cookies.put('grouplist', 'shared', {'path':'/'});
        this.$scope.changePriority = this.changePriority.bind(this);
        this.$scope.onDrop = this.onDrop.bind(this);
    },

    defineListeners: function () {
        this._super()
    },

    changePriority: function (id, priority) {
        this.sharedGroupsService.update(id, {priority}).then(
            () => {
                notify.success("Group's priority changed.");
                this.getGroups()
            },
            error => notify.error(error.data)
        );
    },
    onDrop: (tr, changePriorityFn) =>  {
        const { dataset : { id : currentGroupId, groupPriority: currentPriority }, nextElementSibling : nextGroup, previousElementSibling : previousGroup } = tr;

        let priority = (previousGroup !== null)  ? parseInt(previousGroup.dataset.groupPriority) : parseInt(nextGroup.dataset.groupPriority);

        if (currentPriority > priority && previousGroup) {
            ++priority;
        }

        changePriorityFn(currentGroupId, priority);
    }
});
sharedGroupsListController.$inject = ['$scope', '$routeParams', '$modal', '$cookies', 'sharedGroupsService', 'IEPaginationService', 'personalGroupsService', 'inventoryHelper', 'globalHelper', '$location', '$rootScope'];
