import React from 'react';
import { Box, TextField, InputAdornment, Button } from '@mui/material';
import { Send as SendIcon, StopCircle as StopIcon } from '@mui/icons-material';
import { makeStyles } from 'tss-react/mui';

const useStyles = makeStyles()(theme => ({
  inputContainer: {
    display: 'flex',
    flexDirection: 'column',
    padding: '10px 20px 10px 10px',
    margin: '0 auto',
    width: 675,
    '& input[type="text"]': {
      fontSize: 16,
      color: theme.palette.text.primary,
      fontWeight: 400
    }
  },
  chatInput: {
    flex: 1,
    '.MuiInputBase-root': {
      paddingRight: 0
    },
    'input': {
      padding: '14px 0 14px 8px !important',
      margin: '0px !important'
    }
  }
}));

interface InputAreaProps {
  inputValue: string;
  isLoading: boolean;
  onSendMessage: (message: string) => void;
  onCancelRequest: () => void;
  onInputChange: (value: string) => void;
}

const InputArea: React.FC<InputAreaProps> = ({
  inputValue,
  isLoading,
  onSendMessage,
  onCancelRequest,
  onInputChange
}) => {
  const { classes } = useStyles();

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !isLoading) {
      onSendMessage(inputValue);
    }
  };

  return (
    <Box className={classes.inputContainer}>
      <TextField
        className={classes.chatInput}
        autoComplete="off"
        placeholder="Ask anything about your infrastructure..."
        value={inputValue}
        onChange={e => onInputChange(e.target.value)}
        onKeyDown={handleKeyDown}
        slotProps={{
          input: {
            endAdornment: (
              <InputAdornment position="end">
                {!isLoading ? (
                  <Button
                    disabled={isLoading}
                    color="primary"
                    onClick={() => onSendMessage(inputValue)}
                  >
                    <SendIcon />
                  </Button>
                ) : (
                  <Button
                    disabled={!isLoading}
                    sx={{ filter: 'grayscale(100%)' }}
                    onClick={onCancelRequest}
                  >
                    <StopIcon />
                  </Button>
                )}
              </InputAdornment>
            ),
          },
        }}
      />
    </Box>
  );
};

export default InputArea;
