<?php

namespace FR\V1_0\Controllers;

use AuditLogActions;
use AuditLogObjectTypes;
use AuditLogFields;
use AuditLogService;
use FR\V1_0\ConfigFile\ConfigFile;
use FR\V1_0\Entities\RemoteHub\CreateRemoteHubEntity;
use FR\V1_0\Entities\RemoteHub\RemoteHub;
use FR\V1_0\Entities\RemoteHub\RemoteHubsCollection;
use FR\V1_0\Models\ConfigFileModel;
use FR\V1_0\Models\RemoteHubModel;
use FR\V1_0\Models\SetupHubModel;

/**
 * @uri /fr/v1.0/federation-config
 * @uri /fr/federation-config
 */
class FederationConfig extends \CfProtectedResource
{
    private $remoteHubModel;
    private $setupHubModel;
    private $configFileModel;

    public function __construct($parameters)
    {
        parent::__construct($parameters);

        $cfSettings = \CfSettings::getInstance()->getConnection();
        $cfdb = \CfdbPdo::getInstance()->getConnection();
        $this->remoteHubModel = new RemoteHubModel($cfSettings, $cfdb);
        $this->configFileModel = new ConfigFileModel();
        $this->setupHubModel = new SetupHubModel($cfSettings, $this->username);
    }

    /**
     * @param $request
     *
     * @rbacName Generate federation config file
     * @rbacDescription from information previously stored in database
     * @rbacGroup Federation report API
     * @rbacAlias federationConfig.post
     *
     * @return \Response
     * @throws \Exception
     */
    public function post($request)
    {
        $response = new \Response($request);

        $role = '';
        $this->validate();

        $remoteHubsCollection = new RemoteHubsCollection();
        foreach ($this->remoteHubModel->getList() as $hub) {
            $remoteHub = CreateRemoteHubEntity::createFromDbResult($hub);
            $role = $remoteHub->getRole() == RemoteHub::FEEDER_ROLE ? RemoteHub::SUPERHUB_ROLE : RemoteHub::FEEDER_ROLE;
            $remoteHubsCollection->addRemoteHub($remoteHub);
        }

        $configEntity = new ConfigFile();
        $configEntity->setRemoteHubs($remoteHubsCollection);
        $configEntity->setRole($role);
        $configEntity->setTargetState($this->setupHubModel->getTargetState());

        $this->configFileModel->save(json_encode($configEntity, JSON_PRETTY_PRINT));
        $response->code = \Response::ACCEPTED;

        return $response;
    }

    /**
     * @param $request
     *
     * @rbacName Delete federation config file
     * @rbacGroup Federation report API
     * @rbacAlias federationConfig.delete
     *
     * @return \Response
     */
    public function delete($request)
    {
        $response = new \Response($request);

        $this->validate();
        $this->configFileModel->delete();
        $response->code = \Response::NOCONTENT;

        AuditLogService::register([
            AuditLogFields::ACTOR => $this->username,
            AuditLogFields::OBJECT_TYPE => AuditLogObjectTypes::FEDERATED_REPORTING,
            AuditLogFields::ACTION => AuditLogActions::DELETE,
            AuditLogFields::DETAILS => ['Deleted federation config file.']
        ]);

        return $response;
    }

    /**
     * @throws \ResponseException
     */
    private function validate()
    {
        if ($this->setupHubModel->isConfigured() == false) {
            throw new \ResponseException('You cannot perform this action, hub is not enabled for Federated Reporting.', \Response::BADREQUEST);
        }
    }

}
