<?php

class SharedHostGroupEntity extends HostGroupEntity
{
    private string $creator;
    private int|null $priority = null;

    public const PRIORITY_COLUMN_NAME = 'priority';
    public const FILTER_COLUMN_NAME = 'filter';

    public function __construct(
        string                $creator = '',
        string                $name = '',
        int|null              $priority = null,
        stdClass|array|string $filter = new stdClass(),
        string|null           $description = '',
        ...$args
    ) {
        parent::__construct();
        !empty($creator) && $this->setCreator($creator);
        !empty($name) && $this->setName($name);
        !empty($description) && $this->setDescription($description);
        !empty($priority) && $this->setPriority($priority);
        !empty((array)$filter) && $this->setFilter($filter);
    }

    /**
     * @return int|null
     */
    public function getPriority(): ?int
    {
        return $this->priority;
    }

    /**
     * @param int|null $priority
     */
    public function setPriority(?int $priority): void
    {
        $this->changedItems[] = 'priority';
        $this->priority = $priority;
    }


    /**
     * @return string
     */
    public function getCreator(): string
    {
        return $this->creator;
    }

    /**
     * @param string $creator
     */
    public function setCreator(string $creator): void
    {
        $this->changedItems[] = 'creator';
        $this->creator = $creator;
    }

    public function toArray(): array
    {
        return [
            'name' => $this->getName(),
            'description' => $this->getDescription(),
            'filter' => $this->getFilter(),
            'filter_sql' => $this->getFilterSql(),
            'creator' => $this->getCreator(),
            'priority' => $this->getPriority(),
        ];
    }

    public function validate(): bool|InvalidArgumentException
    {
        if (empty($this->getCreator())) {
            throw new InvalidArgumentException('Group creator cannot be empty.');
        }

        Validator::name($this->getName());
        return true;
    }

    public static function sortableColumns(): array
    {
        return ['name', BaseHostGroupsModel::NAME_INT_SORT_COLUMN, 'id', 'priority', 'creator', 'creation_time'];
    }
}
