import { createAsyncThunk } from '@reduxjs/toolkit';

import Api from '@northern.tech/store/api/general-api';

import { selectCmdbServiceUrl } from './selectors';

export interface CmdbSubEntry {
  id: number;
  hostkey: string;
  entry_id: number;
  item_name: string;
  item_value: string | unknown[] | null;
  item_type: string;
}

export interface CmdbItem {
  id: number;
  hostkey: string;
  created_at: string;
  name: string | null;
  description: string;
  tags: string[];
  type: string;
  meta: Record<string, unknown>;
  entries: CmdbSubEntry[];
}

interface CmdbItemParams {
  identifier: string;
  type: string;
  name: string;
}

interface CreateUpdateItemParams {
  identifier: string;
  id?: string;
  entry: CmdbItem;
  baseUrl?: string;
}

interface GetItemsParams {
  identifier: string;
  baseUrl?: string;
  page?: number;
  perPage?: number;
}

interface DeleteItemParams {
  id: number;
  baseUrl?: string;
}

interface GetJsonConfigParams {
  identifier: string;
  baseUrl?: string;
}

const getModuleUrl = (baseUrl: string | undefined, state: unknown): string => {
  return baseUrl || selectCmdbServiceUrl(state as any) || '';
};

export const getItems = createAsyncThunk(
  'cmdb/getItems',
  async ({ identifier, baseUrl, page = 1, perPage = 6 }: GetItemsParams, { getState }) => {
    const moduleUrl = getModuleUrl(baseUrl, getState());
    const { data } = await Api.get(`${moduleUrl}/items/${identifier}?page=${page}&perPage=${perPage}`);
    return data;
  }
);

export const getItem = createAsyncThunk(
  'cmdb/getItem',
  async (
    { identifier, type, name, baseUrl }: CmdbItemParams & { baseUrl?: string },
    { getState }
  ) => {
    const moduleUrl = getModuleUrl(baseUrl, getState());
    const url = `${moduleUrl}/item/${identifier}/${type}/${name}/`;

    const { data } = await Api.get(url);
    return data;
  }
);

export const getPolicyConfigurationsIds = createAsyncThunk(
  'cmdb/getPolicyConfigurationsIds',
  async (
    { identifier, baseUrl }: { identifier: string; baseUrl?: string },
    { getState }
  ) => {
    const moduleUrl = getModuleUrl(baseUrl, getState());
    const { data } = await Api.get(`${moduleUrl}/policy_configurations_ids/${identifier}`);
    return data;
  }
);

export const createItem = createAsyncThunk(
  'cmdb/createItem',
  async ({ identifier, entry, baseUrl }: CreateUpdateItemParams, { getState, rejectWithValue }) => {
    try {
      const moduleUrl = getModuleUrl(baseUrl, getState());

      const formData = new URLSearchParams();
      formData.append('data', JSON.stringify(entry));

      await Api.post(`${moduleUrl}/item/${identifier}`, formData.toString(), {
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
        },
      });

      notify.success('Host specific data successfully created.');
      return entry;
    } catch (error) {
      notify.error('Failed to create host specific data. Please check the application logs.');
      return rejectWithValue(error);
    }
  }
);

export const updateItem = createAsyncThunk(
  'cmdb/updateItem',
  async ({ id, entry, baseUrl }: CreateUpdateItemParams & { id: string }, { getState, rejectWithValue }) => {
    const moduleUrl = getModuleUrl(baseUrl, getState());

    const formData = new URLSearchParams();
    formData.append('data', JSON.stringify(entry));
    try {
      await Api.put(`${moduleUrl}/item/${id}/`, formData.toString(), {
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
        },
      });
      notify.success(`Host specific data succesfully updated.`)

      return { id };
    } catch (error) {
      notify.error('Failed to create host specific data. Please check the application logs.');
      return rejectWithValue(error);
    }
  }
);

export const deleteItem = createAsyncThunk(
  'cmdb/deleteItem',
  async ({ id, baseUrl }: DeleteItemParams, { getState }) => {
    const moduleUrl = getModuleUrl(baseUrl, getState());
    await Api.delete(`${moduleUrl}/item/${id}/`, {});
    notify.success('Host specific data successfully deleted.');
    return { id };
  }
);

export const getJSONConfig = createAsyncThunk(
  'cmdb/getJSONConfig',
  async ({ identifier, baseUrl }: GetJsonConfigParams, { getState }) => {
    const moduleUrl = getModuleUrl(baseUrl, getState());
    const { data } = await Api.get(`${moduleUrl}/json/${identifier}/`);
    return data;
  }
);

interface AutocompleteParams {
  query: string;
  baseUrl?: string;
}

export const getAutocompleteClasses = createAsyncThunk(
  'autocomplete/getClasses',
  async ({ query, baseUrl = '/autocomplete' }: AutocompleteParams) => {
    const { data } = await Api.get(`${baseUrl}/classes?query=${encodeURIComponent(query)}`);
    return data;
  }
);

export const getAutocompleteVariables = createAsyncThunk(
  'autocomplete/getVariables',
  async ({ query, baseUrl = '/autocomplete' }: AutocompleteParams) => {
    const { data } = await Api.get(`${baseUrl}/variables?query=${encodeURIComponent(query)}`);
    return data;
  }
);

export const getAutocompleteInventory = createAsyncThunk(
  'autocomplete/getInventory',
  async ({ query, baseUrl = '/autocomplete' }: AutocompleteParams) => {
    const { data } = await Api.get(`${baseUrl}/inventory?query=${encodeURIComponent(query)}`);
    return data;
  }
);
